package com.reyun.service.impl;

import com.reyun.dic.DownloadStatusEnumType;
import com.reyun.model.Account;
import com.reyun.model.App;
import com.reyun.model.ExportReportInfo;
import com.reyun.repository.AppRepository;
import com.reyun.repository.ExportReportInfoRepository;
import com.reyun.service.ExportReportInfoService;
import com.reyun.util.AwsS3Util;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

@Service
public class ExportReportInfoServiceImpl implements ExportReportInfoService {

    @Autowired
    ExportReportInfoRepository repository;

    @Autowired
    AppRepository appRepository;

    private final static int EXPORT_INVALID_DAYS = 3;
    private final static int EXPORT_INVALID_MONTH = 1;

    private static final String S3_DOWNLOAD_BUCKET = "reyundownload";

    private static final String DOWNLOAD_PATH = "trackingio/reportdownload/";

    @Override
    public ExportReportInfo save(ExportReportInfo exportReport) {
        exportReport.setCreateTime(new Date());
        return repository.save(exportReport);
    }

    @Override
    public void delete(Long id) {
        ExportReportInfo reportInfo = repository.findOne(id);
        final String s3key = DOWNLOAD_PATH + reportInfo.getFileName();
        //删除文件
        if (AwsS3Util.getInstance().getAmazonS3().doesObjectExist(S3_DOWNLOAD_BUCKET, s3key)) {
            AwsS3Util.getInstance().deleteS3Object(S3_DOWNLOAD_BUCKET, s3key);
        }
        //删除记录
        repository.delete(id);

    }

    @Override
    public List<ExportReportInfo> findByApp(Account loginAccount, Long app) {

        Date startDate = new DateTime().minusMonths(EXPORT_INVALID_MONTH).toDate();

        //母账号查看所有导出报表,子账号只能查看自己创建的 add by sunhao 20170828
        if (loginAccount.getIsSuperUser()) {

            return repository.findByApp(app, startDate);

        } else {

            return repository.findByAccountAndApp(app, startDate, loginAccount.getId());
        }
    }

    @Override
    public boolean validFileName(ExportReportInfo exportReport) {
        App app = appRepository.findOne(exportReport.getApp());
        String fileName = app.getAppkey() + "_" + exportReport.getReportName() + "_" + exportReport.getStartDate() + "_" + exportReport.getEndDate() + ".txt";
        ExportReportInfo export = repository.findByAppAndFileName(exportReport.getApp(), fileName);
        if (export == null) {
            return true;
        }
        return false;
    }

    @Override
    public int updateStatus(Long id, String status, Long account) {
        return repository.updateStatus(id, status, new Date(), account);
    }

    @Override
    public int updateStatus(Long id, String status) {
        return repository.updateStatus(id, status, new Date());
    }

    @Override
    public ExportReportInfo findOne(Long id) {
        return repository.findOne(id);
    }

    @Override
    public List<ExportReportInfo> findExport2Deal() {
        return repository.findByStatus(DownloadStatusEnumType.INIT.getCode(), DownloadStatusEnumType.FAILED.getCode());
    }

    @Override
    public List<ExportReportInfo> findInvalidExportList() {

        //获取三天前的日期
        Date threeDaysAgo = DateTime.now().minusDays(EXPORT_INVALID_DAYS).toDate();
        //查找三天前的已生成的数据
        return repository.listInvalidExport(threeDaysAgo);
    }

    /**
     * 更新下载信息
     */
    @Override
    public int updateDownloadInfo(Long downloadId, Account account, String downloadIp) {

        return repository.updateDownloadInfo(downloadId, account.getId(), account.getName(), downloadIp, new Date());
    }
}
