package mobvista.dmp.datasource.age.mapreduce;

import com.google.common.collect.Sets;
import mobvista.dmp.util.MRUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.compress.GzipCodec;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.lib.input.FileInputFormat;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.util.GenericOptionsParser;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.type.JavaType;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Created by liushauo on 2017/2/16 0016.
 * desc:清洗每日dsp数据，得到年龄
 */
public class GetDspAgeMR {
    public static void main(String[] args) throws ClassNotFoundException, IOException, InterruptedException {
        Configuration conf = new Configuration();
        String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
        conf.set("now", otherArgs[2]);
        conf.set("mapreduce.map.memory.mb", "1433");
        conf.set("mapreduce.map.speculative", "true");
        conf.set("mapreduce.reduce.speculative", "true");
        conf.set("mapreduce.task.io.sort.mb", "500");
        conf.set("mapreduce.reduce.java.opts", "-Xmx1536m");
        conf.set("mapreduce.reduce.memory.mb", "2048");
        conf.set("mapreduce.reduce.shuffle.parallelcopies", "50");
        Job job = Job.getInstance(conf, "DMP-GetDspAgeMR-fengliang");
        FileOutputFormat.setCompressOutput(job, true);
        FileOutputFormat.setOutputCompressorClass(job, GzipCodec.class);
        job.setJarByClass(GetDspAgeMR.class);
        job.setMapperClass(GetDspAgeMapper.class);
        job.setOutputKeyClass(Text.class);
        job.setOutputValueClass(Text.class);
        job.setInputFormatClass(TextInputFormat.class);
        FileInputFormat.addInputPath(job, new Path(otherArgs[0]));
        FileOutputFormat.setOutputPath(job, new Path(otherArgs[1]));
        System.exit(job.waitForCompletion(true) ? 0 : 1);
    }

    public static class GetDspAgeMapper extends Mapper<Object, Text, Text, Text> {
        ObjectMapper objectMapper = new ObjectMapper();
        JavaType mapType = objectMapper.getTypeFactory().constructMapType(HashMap.class, String.class, String.class);
        Text outKey = new Text();
        Text outValue = new Text();
        int now;
        Set<String> matchingSet = Sets.newHashSet("","0","1970","GB","null","-");

        public void setup(Context context) throws IOException, InterruptedException {
            now = Integer.parseInt(context.getConfiguration().get("now"));
        }

        public void map(Object key, Text value, Context context) throws IOException, InterruptedException {
            String line = value.toString();
            try {
                Map<String, String> dspMap = objectMapper.readValue(line, mapType);
                String deviceId = dspMap.get("device_id");
                if (deviceId == null || deviceId.equals("")) {
                    return;
                }
                String[] tmpDeviceId = Util.lineSplit.split(deviceId, -1);
                if (tmpDeviceId.length != 5) {
                    return;
                }
                if (Util.match.matcher(deviceId).matches()) {
                    return;
                }
                String birthday = dspMap.get("birthday");
                if (matchingSet.contains(birthday)) {//输出有年龄的
                    return;
                }
                int age = now - Integer.parseInt(birthday);
                if (age >= 100 || age <= 0) {
                    return;
                }
                int label = Util.calcLabel(age);

                String deviceType = dspMap.get("device_type");
                outKey.set(deviceId);
                outValue.set(MRUtils.JOINER.join("A", label, "dsp", deviceType));
                context.write(outKey, outValue);
            } catch (Exception e) {
                return;
            }
        }
    }
}
