package mobvista.dmp.datasource.adn

import mobvista.dmp.common.{CommonSparkJob, MobvistaConstant}
import mobvista.dmp.util.MRUtils
import org.apache.commons.cli.{BasicParser, Options}
import org.apache.hadoop.fs.{FileSystem, Path}
import org.apache.hadoop.io.compress.GzipCodec

import java.net.URI
import scala.collection.mutable.ArrayBuffer

/**
 * @package: mobvista.dmp.datasource.adn
 * @author: wangjf
 * @date: 2021/2/23
 * @time: 5:12 下午
 * @email: jinfeng.wang@mobvista.com
 */
class AdnClickJoinInstallDaily extends CommonSparkJob with Serializable {
  def commandOptions(): Options = {
    val options = new Options()
    options.addOption("input_click", true, "input_click")
    options.addOption("input_install", true, "input_install")
    options.addOption("output", true, "output")
    options.addOption("coalesce", true, "coalesce")
    options
  }

  override protected def run(args: Array[String]): Int = {
    val parser = new BasicParser()
    val options = commandOptions()
    val commandLine = parser.parse(options, args)

    val input_click = commandLine.getOptionValue("input_click")
    val input_install = commandLine.getOptionValue("input_install")
    val output = commandLine.getOptionValue("output")
    val coalesce = commandLine.getOptionValue("coalesce")

    val spark = MobvistaConstant.createSparkSession(s"AdnClickJoinInstallDaily")

    try {
      val sc = spark.sparkContext
      //  deviceId, deviceType, requestId, campaignId, platform
      val clickRdd = sc.textFile(input_click).map(_.split("\t")).map(r => {
        val deviceId = r(0)
        val deviceType = r(1)
        val requestId = r(2)
        val campaignId = r(3)
        val platform = r(4)
        (MRUtils.JOINER.join(requestId, campaignId), (deviceId, deviceType, platform))
      }).combineByKey(
        (v: (String, String, String)) => Iterable(v),
        (c: Iterable[(String, String, String)], v: (String, String, String)) => c ++ Seq(v),
        (c1: Iterable[(String, String, String)], c2: Iterable[(String, String, String)]) => c1 ++ c2
      )
      //  deviceId, deviceType, platform, requestId, campaignId, packageName
      val installRdd = sc.textFile(input_install).map(_.split("\t")).map(r => {
        val deviceId = r(0)
        val deviceType = r(1)
        val platform = r(2)
        val requestId = r(3)
        val campaignId = r(4)
        val packageName = r(5)
        (MRUtils.JOINER.join(requestId, campaignId), (deviceId, deviceType, platform, packageName))
      }).combineByKey(
        (v: (String, String, String, String)) => Iterable(v),
        (c: Iterable[(String, String, String, String)], v: (String, String, String, String)) => c ++ Seq(v),
        (c1: Iterable[(String, String, String, String)], c2: Iterable[(String, String, String, String)]) => c1 ++ c2
      )

      val rdd = installRdd.join(clickRdd).map(rs => {
        val arrayBuffer = new ArrayBuffer[String]
        val keys = MRUtils.SPLITTER.split(rs._1, -1)
        val campaignId = keys(1)
        val install = rs._2._1
        //  val click = rs._2._2
        install.foreach(e => {
          val deviceId = e._1
          val deviceType = e._2
          val platform = e._3
          val packageName = e._4
          arrayBuffer += MRUtils.JOINER.join(deviceId, deviceType, platform, campaignId, packageName)
        })
        arrayBuffer.toIterator
      }).flatMap(l => l)

      FileSystem.get(new URI(s"s3://mob-emr-test"), spark.sparkContext.hadoopConfiguration).delete(new Path(output), true)
      rdd.repartition(coalesce.toInt).saveAsTextFile(output, classOf[GzipCodec])
    } finally {
      if (spark != null) {
        spark.stop()
      }
    }
    0
  }

}

object AdnClickJoinInstallDaily {
  def main(args: Array[String]): Unit = {
    new AdnClickJoinInstallDaily().run(args)
  }
}