package mobvista.dmp.datasource.datatory

import java.net.URI

import mobvista.dmp.common.CommonSparkJob
import org.apache.commons.cli.{BasicParser, Options}
import org.apache.commons.lang3.StringUtils
import org.apache.hadoop.fs.{FileSystem, Path}
import org.apache.spark.sql.types.{StringType, StructField, StructType}
import org.apache.spark.sql.{Dataset, Row, SaveMode, SparkSession}
import org.apache.spark.storage.StorageLevel

/**
  * @package: mobvista.dmp.datasource.datatory
  * @author: wangjf
  * @date: 2019/4/25
  * @time: 下午3:19
  * @email: jinfeng.wang@mobvista.com
  * @phone: 152-1062-7698
  */
class TrackingEventDaily extends CommonSparkJob with java.io.Serializable {
  def commandOptions(): Options = {
    val options = new Options()
    options.addOption("date", true, "date")
    options.addOption("before_date", true, "before_date")
    options.addOption("output", true, "output")
    options.addOption("info_output", true, "info_output")
    options
  }

  override protected def run(args: Array[String]): Int = {
    val parser = new BasicParser()
    val options = commandOptions()
    val commandLine = parser.parse(options, args)
    val date = commandLine.getOptionValue("date")
    val before_date = commandLine.getOptionValue("before_date")
    val output = commandLine.getOptionValue("output")
    val info_output = commandLine.getOptionValue("info_output")

    val spark = SparkSession
      .builder()
      .appName("TrackingEventDaily")
      .config("spark.rdd.compress", "true")
      .config("spark.shuffle.compress", "true")
      .config("spark.sql.orc.filterPushdown", "true")
      .config("spark.io.compression.codec", "lz4")
      .config("spark.io.compression.lz4.blockSize", "64k")
      .config("spark.sql.warehouse.dir", "s3://mob-emr-test/spark-warehouse")
      .config("spark.serializer", "org.apache.spark.serializer.KryoSerializer")
      .enableHiveSupport()
      .getOrCreate()

    val sc = spark.sparkContext
    try {
      import spark.implicits._

      /*
      campaingMaps = sc.broadcast(Constant.jdbcConnection(spark, "mob_adn", "campaign_list")
        .select("id", "network_cid")
        .groupBy("network_cid")
        .agg(concat_ws(",", collect_set("id")))
        .filter(r => {
          StringUtils.isNotBlank(r.getAs("network_cid").toString)
        }).rdd.map(r => {
        (r.getAs("network_cid").toString, r.getAs("id").toString)
      }).collectAsMap())

      spark.udf.register("getCampaignIds", getCampaignIds _)

      Constant.jdbcConnection(spark, "mob_adn", "campaign_list")
        .rdd.map(r => {
        (r.getAs("network_cid").toString, r.getAs("id"))
      }).saveAsTextFile(output, classOf[GzipCodec])
      */

      FileSystem.get(new URI(s"s3://mob-emr-test"), sc.hadoopConfiguration).delete(new Path(output), true)

      var sql = Constant.tracking_event_sql.replace("@date", date)

      spark.sql(sql)
        .filter(r => {
          Constant.check_deviceId(r.getAs("device_id"))
        })
        .write
        .mode(SaveMode.Overwrite)
        .option("orc.compress", "snappy")
        .orc(output)

      val campaingDF = Constant.jdbcConnection(spark, "mob_adn", "campaign_list")
        .select("id", "network_cid")
        .filter(r => {
          StringUtils.isNotBlank(r.getAs("network_cid").toString)
        }).rdd.map(r => {
        Row(r.getAs("network_cid").toString, r.getAs("id").toString)
      })
      val schema = StructType(Array(
        StructField("network_cid", StringType),
        StructField("id", StringType)))
      spark.createDataFrame(campaingDF, schema = schema).createOrReplaceTempView("campaign_tb")

      sql = Constant.event_info_sql.replace("@date", date)
      val df_1 = spark.sql(sql)
      sql = Constant.tracking_event_info_sql.replace("@date", date)
      val df_2 = spark.sql(sql)

      val df: Dataset[Row] = df_1.union(df_2)
        .dropDuplicates
        .rdd
        .filter(r => {
          r.getAs("offer_id") != null && StringUtils.isNotBlank(r.getAs("offer_id").toString)
        })
        .map(r => {
          EventInfo(r.getAs("id"), r.getAs("event_name"), r.getAs("event_type"), r.getAs("offer_id"))
        }).toDF

      val oldDF = spark.sql(Constant.all_event_info_sql.replace("@before_date", before_date))

      val newDF = oldDF.union(df).dropDuplicates.repartition(1).persist(StorageLevel.MEMORY_ONLY_SER)
      newDF
        .write
        .mode(SaveMode.Overwrite)
        .option("orc.compress", "snappy")
        .orc(info_output)

      Constant.writeMySQL(newDF, "event_info", SaveMode.Overwrite)
    } finally {
      if (spark != null) {
        spark.stop()
      }
    }
    0
  }

  /*
  def getCampaignIds(id: String): String = {
    campaingMaps.value.getOrElse(id, "")
  }
  */
}

object TrackingEventDaily {
  def main(args: Array[String]): Unit = {
    new TrackingEventDaily().run(args)
  }
}