package mobvista.dmp.datasource.dm

import java.net.URI
import java.text.SimpleDateFormat

import mobvista.dmp.common.CommonSparkJob
import mobvista.dmp.util.DateUtil
import org.apache.commons.cli.{BasicParser, Options}
import org.apache.hadoop.fs.{FileSystem, Path}
import org.apache.spark.sql.{SaveMode, SparkSession}

/**
  * @package: mobvista.dmp.datasource.dm
  * @author: wangjf
  * @date: 2019/3/18
  * @time: 上午11:16
  * @email: jinfeng.wang@mobvista.com
  * @phone: 152-1062-7698
  */
class ActiveTag extends CommonSparkJob with Serializable {
  def commandOptions(): Options = {
    val options = new Options()
    options.addOption("date", true, "date")
    options.addOption("days", true, "days")
    options.addOption("output", true, "output")
    options.addOption("coalesce", true, "coalesce")
    options
  }

  override protected def run(args: Array[String]): Int = {
    val parser = new BasicParser()
    val options = commandOptions()
    val commandLine = parser.parse(options, args)
    val date = commandLine.getOptionValue("date")
    val days = commandLine.getOptionValue("days").toInt
    val output = commandLine.getOptionValue("output")
    val coalesce = commandLine.getOptionValue("coalesce")

    val spark = SparkSession
      .builder()
      .appName(s"ActiveTag.${days}")
      .config("spark.rdd.compress", "true")
      .config("spark.shuffle.compress", "true")
      .config("spark.sql.orc.filterPushdown", "true")
      .config("spark.io.compression.codec", "lz4")
      .config("spark.io.compression.lz4.blockSize", "64k")
      .config("spark.sql.warehouse.dir", "s3://mob-emr-test/spark-warehouse")
      .config("spark.serializer", "org.apache.spark.serializer.KryoSerializer")
      .enableHiveSupport()
      .getOrCreate()

    FileSystem.get(new URI(s"s3://mob-emr-test"), spark.sparkContext.hadoopConfiguration).delete(new Path(output), true)

    try {

      spark.udf.register("check_deviceId", Constant.check_deviceId _)
      spark.udf.register("combineJson", Constant.combineJson _)

      val beforeDays = DateUtil.getDayByString(date, "yyyyMMdd", -days)

      val active_sql = Constant.active_sql.replace("@date", date).replace("@before_days", beforeDays)
        .replace("@check_deviceId", "check_deviceId(device_id)")
        .replace("@combineJson", "combineJson")

      spark.sql(active_sql).createOrReplaceTempView("active")

      val sdf1 = new SimpleDateFormat("yyyy-MM-dd")
      val sdf2 = new SimpleDateFormat("yyyyMMdd")
      val date2 = sdf1.format(sdf2.parse(date))
      val beforeDays2 = sdf1.format(sdf2.parse(beforeDays))

      val user_info_sql = Constant.user_info_sql.replace("@date", date)
        .replace("@date2", date2).replace("@before_days2", beforeDays2)

      spark.sql(user_info_sql).createOrReplaceTempView("info")

      val region_sql = Constant.region_sql.replace("@date", date)
      spark.sql(region_sql).createOrReplaceTempView("region")

      spark.sql(Constant.active_join_info_sql)
        .dropDuplicates()
        .repartition(coalesce.toInt)
        .write
        .mode(SaveMode.Overwrite)
        .option("orc.compress", "snappy")
        .orc(output)

    } finally {
      if (spark != null) {
        spark.stop()
      }
    }
    0
  }
}

object ActiveTag {
  def main(args: Array[String]): Unit = {
    new ActiveTag().run(args)
  }
}
